/* TID parsing for GDB, the GNU debugger.

   Copyright (C) 2015-2024 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include "tid-parse.h"
#include "inferior.h"
#include "gdbthread.h"
#include <ctype.h>

/* See tid-parse.h.  */

[[noreturn]] void
invalid_thread_id_error (const char *string)
{
  error (_("Invalid thread ID: %s"), string);
}

/* Wrapper for get_number_trailer that throws an error if we get back
   a negative number.  We'll see a negative value if the number is
   stored in a negative convenience variable (e.g., $minus_one = -1).
   STRING is the parser string to be used in the error message if we
   do get back a negative number.  */

static bool
get_non_negative_number_trailer (const char **pp, int *parsed_value,
				 int trailer, const char *string)
{
  bool retval = get_number_trailer (pp, parsed_value, trailer);

  if (*parsed_value < 0)
    error (_("negative value: %s"), string);

  return retval;
}

/* Parse TIDSTR as a per-inferior lane ID, in one of the following forms:

     INF_NUM.THR_NUM:LANE_NUM, THR_NUM:LANE, THR_NUM, :LANE_NUM

   Return a tuple; the first item of the tuple is INF_NUM, the second item
   is THR_NUM, and the third is LANE_NUM.

   If TIDSTR does not include an INF_NUM component, then the first item in
   the tuple will be 0 (which is an invalid inferior number), this indicates
   that TIDSTR references the current inferior.

   If TIDSTR does not include a THR_NUM component, then the second item in
   the tuple will be 0 (which is an invalid thread number).  This indicates
   that TIDSTR references the current thread.

   If TIDSTR does not include a LANE_NUM component, then the third item in
   the tuple will be -1 (which is an invalid lane number).  This indicates
   that TIDSTR references the default lane.

   This function does not validate the INF_NUM, THR_NUM and LANE_NUM are
   actually valid numbers, that is, they might reference inferiors, threads
   or lanes that don't actually exist; this function just splits the string
   into its component parts.

   If there is an error parsing TIDSTR then this function will raise an
   exception.  */

static std::tuple<int, int, int>
parse_thread_id_1 (const char *tidstr, const char **end)
{
  const char *number = tidstr;
  const char *dot, *p1;
  int thr_num, inf_num, lane_num;

  dot = strchr (number, '.');

  if (dot != NULL)
    {
      /* Parse number to the left of the dot.  */
      p1 = number;
      if (!get_non_negative_number_trailer (&p1, &inf_num, '.', number))
	invalid_thread_id_error (number);

      if (inf_num == 0)
	invalid_thread_id_error (number);
      p1 = dot + 1;
    }
  else
    {
      inf_num = 0;
      p1 = number;
    }

  dot = strchr (p1, ':');
  bool lane_specified = (dot != nullptr);
  char trailer = lane_specified ? ':' : 0;

  if (p1[0] != ':')
    {
      /* Thread number is presented.  */
      if (!get_non_negative_number_trailer (&p1, &thr_num, trailer, number))
	invalid_thread_id_error (number);

      if (thr_num == 0)
	invalid_thread_id_error (number);
    }
  else
    {
      /* Only lane number is specified.  Reference the current thread.  */
      thr_num = 0;
    }

  lane_num = -1;
  if (lane_specified)
    {
      p1 = dot + 1;

      if (!get_non_negative_number_trailer (&p1, &lane_num, 0, dot))
	error (_("Incorrect SIMD lane number: %s."), dot);

      if (lane_num >= (sizeof (unsigned int)) * 8)
	error (_("Incorrect SIMD lane number: %d."), lane_num);
    }

  if (end != nullptr)
    *end = p1;

  return { inf_num, thr_num, lane_num };
}

/* See tid-parse.h.  */

struct thread_info *
parse_thread_id (const char *tidstr, const char **end, int *simd_lane_num)
{
  const auto [inf_num, thr_num, lane_num]
    = parse_thread_id_1 (tidstr, end);

  inferior *inf;
  bool explicit_inf_id = false;

  if (inf_num != 0)
    {
      inf = find_inferior_id (inf_num);
      if (inf == nullptr)
	error (_("No inferior number '%d'"), inf_num);
      explicit_inf_id = true;
    }
  else
    inf = current_inferior ();

  thread_info *tp = nullptr;
  if (thr_num == 0)
    {
      /* Take the current thread.  */
      if (inferior_ptid == null_ptid)
	error (_("No thread selected."));
      tp = inferior_thread ();
    }
  else
    {
      for (thread_info *it : inf->threads ())
	if (it->per_inf_num == thr_num)
	  {
	    tp = it;
	    break;
	  }

      if (tp == nullptr)
	{
	  if (show_inferior_qualified_tids () || explicit_inf_id)
	    error (_("Unknown thread %d.%d."), inf->num, thr_num);
	  else
	    error (_("Unknown thread %d."), thr_num);
	}
    }

  if (lane_num != -1)
    {
      if (tp->executing ())
	error (_("Thread %s is executing, cannot check SIMD lane status:"
		 " Cannot specify SIMD lane"), print_thread_id (tp));

      if (!target_has_registers ())
	error (_("Target of thread %s has no registers, cannot check SIMD lane"
		 " status: Cannot specify SIMD lane"), print_thread_id (tp));

      if (!tp->has_simd_lanes ())
	error (_("Thread %s does not have SIMD lanes: Cannot specify SIMD"
		 " lane"), print_thread_id (tp));

      if (simd_lane_num == nullptr)
	error (_("SIMD lane is not supported."));
    }

  if (simd_lane_num != nullptr)
    *simd_lane_num = lane_num;

  return tp;
}

/* See tid-parse.h.  */

bool
is_thread_id (const char *tidstr, const char **end)
{
  try
    {
      (void) parse_thread_id_1 (tidstr, end);
      return true;
    }
  catch (const gdb_exception_error &)
    {
      return false;
    }
}

/* See tid-parse.h.  */

tid_range_parser::tid_range_parser (const char *tidlist,
				    int default_inferior,
				    int default_thr_num)
{
  init (tidlist, default_inferior, default_thr_num);
}

/* See tid-parse.h.  */

void
tid_range_parser::init (const char *tidlist,
			int default_inferior,
			int default_thr_num)
{
  m_state = STATE_INFERIOR;
  m_cur_tok = tidlist;
  m_inf_num = 0;
  m_thr_num = 0;
  m_simd_lane_num = -1;
  m_qualified = false;
  m_default_inferior = default_inferior;
  m_default_thr_num = default_thr_num;
  m_in_thread_star_range = false;
  m_in_simd_lane_star_range = false;
}

/* This function returns true only if CUR_TOK is immediately followed by a
   logical operator.  Because a CUR_TOK value, starting with "$" and
   followed by a logical operator is considered as an expression.  Otherwise,
   it returns false and the parser will continue to parse the thread ids.
   These are the input examples:

   thread filter $ONE-$TWO $_VAR>0
   thread filter $ONE-$TWO $_VAR > 0
   thread filter 1 2 3 $VAR==2
*/

static bool
is_filter_expression (const char *cur_tok)
{
  const char *p = cur_tok;
  const char *space = skip_to_space (cur_tok);

  /* If input contains logical operator before the first space then
     end the parser.  */
  while (p < space && *p != ' ' && !(THREAD_FILTER_IS_LOGICAL_OP (*p)))
    p++;

  if (THREAD_FILTER_IS_LOGICAL_OP (*p))
    return true;

  /* Skip space if the expression itself contains space after the
     convenience variable in expression.  */
  const char *s = skip_spaces (p);
  if (*s != '\0' && THREAD_FILTER_IS_LOGICAL_OP (*s))
    return true;

  return false;
}

/* See tid-parse.h.  */

bool
tid_range_parser::finished () const
{
  switch (m_state)
    {
    case STATE_INFERIOR:
      /* Parsing is finished when at end of string or null string,
	 or we are not in a range or if $ is used in an expression and
	 not in front of an integer, negative integer, convenience var
	 or negative convenience var.  */
      return (*m_cur_tok == '\0'
	      || !(isdigit (*m_cur_tok)
		   || (*m_cur_tok == '$' && !is_filter_expression (m_cur_tok))
		   || *m_cur_tok == '*'
		   || *m_cur_tok == ':'
		   || *m_cur_tok == '['));
    case STATE_THREAD_RANGE:
      return m_range_parser.finished ();
    case STATE_SIMD_LANE_RANGE:
      return m_simd_lane_range_parser.finished ();
    }

  gdb_assert_not_reached ("unhandled state");
}

/* See tid-parse.h.  */

const char *
tid_range_parser::cur_tok () const
{
  switch (m_state)
    {
    case STATE_INFERIOR:
      return m_cur_tok;
    case STATE_THREAD_RANGE:
      return m_range_parser.cur_tok ();
    case STATE_SIMD_LANE_RANGE:
      return m_simd_lane_range_parser.cur_tok ();
    }

  gdb_assert_not_reached ("unhandled state");
}

/* See tid-parse.h.  */

bool
tid_range_parser::in_thread_state () const {
  return m_state == STATE_THREAD_RANGE;
}

/* See tid-parse.h.  */

bool
tid_range_parser::in_simd_lane_state () const
{
  return m_state == STATE_SIMD_LANE_RANGE;
}

/* See tid-parse.h.  */

void
tid_range_parser::skip_range ()
{
  gdb_assert (in_thread_state () || in_simd_lane_state ());

  if (m_range_parser.in_range ())
    m_range_parser.skip_range ();

  if (m_simd_lane_range_parser.in_range ())
    m_simd_lane_range_parser.skip_range ();

  /* Advance to next item, either a command or the next tid.  Step over
     spaces inside square brackets.  */
  const char *cur_tok = skip_to_next (m_cur_tok);

  init (cur_tok, m_default_inferior, m_default_thr_num);
}

/* See tid-parse.h.  */

void
tid_range_parser::skip_simd_lane_range ()
{
  gdb_assert (in_simd_lane_state ());
  if (m_simd_lane_range_parser.in_range ())
    m_simd_lane_range_parser.skip_range ();
  else if (m_simd_lane_range_parser.in_set ())
    m_simd_lane_range_parser.skip_set ();

  if (m_range_parser.in_range ())
    {
      /* The thread range was not finished yet.  */
      m_state = STATE_THREAD_RANGE;
    }
  else
    {
      /* We are ready to go to the next token in the list (e.g. after
	 a space).  */
      const char *cur_tok = m_simd_lane_range_parser.cur_tok ();
      init (cur_tok, m_default_inferior, m_default_thr_num);
    }
}

/* See tid-parse.h.  */

unsigned int
tid_range_parser::simd_lane_range_end ()
{
    gdb_assert (in_simd_lane_state ());
    return m_simd_lane_range_parser.end_value ();
}

/* See tid-parse.h.  */

bool
tid_range_parser::tid_is_qualified () const
{
  return m_qualified;
}

/* See tid-parse.h.  */

bool
tid_range_parser::process_inferior_state (const char *next)
{
  const char *p = m_cur_tok;

  while (p < next && *p != '.')
    p++;

  if (p < next)
    {
      const char *dot = p;

      /* Parse number to the left of the dot.  */
      p = m_cur_tok;
      if (!get_non_negative_number_trailer (&p, &m_inf_num, '.', m_cur_tok))
	return false;

      if (m_inf_num == 0)
	error (_("Invalid thread ID 0: %s"), m_cur_tok);

      m_qualified = true;
      p = dot + 1;

      if (isspace (*p))
	return false;
    }
  else
    {
      m_inf_num = m_default_inferior;
      m_qualified = false;
      p = m_cur_tok;
    }

  m_range_parser.init (p, ':');

  m_state = STATE_THREAD_RANGE;
  if (p[0] == '*' && (p[1] == '\0' || p[1] == ':' || isspace (p[1])))
    {
      /* Setup the number range parser to return numbers in the
	 whole [1,INT_MAX] range.  */
      m_range_parser.setup_range (1, INT_MAX, skip_spaces (p + 1));
      m_in_thread_star_range = true;
    }
  else
    m_in_thread_star_range = false;

  return true;
}

/* See tid-parse.h.  */

bool
tid_range_parser::process_thread_state (const char *next)
{
  get_number_status thread_parse_error
    = m_range_parser.get_number_overflow (&m_thr_num);

  /* Even if the thread parser failed, we want to check if SIMD lane
     range is specified.  */

  if (thread_parse_error == NUMBER_OK && m_thr_num < 0)
    error (_("negative value: %s"), m_cur_tok);
  else if (thread_parse_error == NUMBER_CONVERSION_ERROR && m_thr_num == 0)
    error (_("Out of bounds value: %s"), m_cur_tok);
  else  if (thread_parse_error == NUMBER_OK && m_thr_num == 0)
    error (_("Invalid thread ID 0: %s"), m_cur_tok);

  const char *colon = strchr (m_cur_tok, ':');

  if (colon != nullptr && colon < next)
    {
      /* A colon is presented in a current token before the expression.
	 That means, that for the current thread range, a SIMD lane
	 range is specified.  */

      m_range_parser.set_end_ptr (colon);

      /* When thread ID is skipped, thread parser returns false.
	 In that case, return the default thread.  */
      if (thread_parse_error != NUMBER_OK && m_cur_tok[0] == ':')
	{
	  m_thr_num = m_default_thr_num;
	  thread_parse_error = NUMBER_OK;
	}

      /* Step over the colon.  */
      colon++;
      m_simd_lane_range_parser.init (colon);
      m_state = STATE_SIMD_LANE_RANGE;

      if (colon[0] == '*' && (colon[1] == '\0' || isspace (colon[1])))
	{
	  m_simd_lane_range_parser.setup_range (0, m_simd_max_len - 1,
						skip_spaces (colon + 1));
	  m_in_simd_lane_star_range = true;
	}
      else
	m_in_simd_lane_star_range = false;
    }

  return (thread_parse_error == NUMBER_OK);
}

/* See tid-parse.h.  */

bool
tid_range_parser::process_simd_lane_state ()
{
  int simd_lane_num;
  get_number_status numerr
      = m_simd_lane_range_parser.get_number_overflow (&simd_lane_num);
  if (NUMBER_ERROR == numerr)
    {
      /* SIMD lanes are specified, but its parsing failed.  */
      m_state = STATE_INFERIOR;
      return false;
    }

  /* Specifying a number that can't be parsed into an integer is bad.  */
  if (NUMBER_CONVERSION_ERROR == numerr)
    error (_("SIMD lane number out of integer bounds."));

  if (simd_lane_num >= m_simd_max_len)
    {
      /* Too large SIMD lane number was specified.  */
      error (_("Incorrect SIMD lane number: %d."), simd_lane_num);
    }

  m_simd_lane_num = simd_lane_num;

  /* If parsing the SIMD lanes is finished, go back to parsing threads.  */
  if (m_simd_lane_range_parser.finished ())
    m_state = STATE_THREAD_RANGE;

  return true;
}

/* Helper for tid_range_parser::get_tid and
   tid_range_parser::get_tid_range.  Return the next range if THR_END
   is non-NULL, return a single thread ID otherwise.  */

bool
tid_range_parser::get_tid_or_range (int *inf_num,
				    int *thr_start, int *thr_end,
				    int *simd_lane_num)
{
  /* Only one out of thr_end and simd_lane is allowed to be specified.  */
  gdb_assert (simd_lane_num == nullptr || thr_end == nullptr);

  const char *next = skip_to_next (m_cur_tok);

  if (m_state == STATE_INFERIOR)
    {
      if (!process_inferior_state (next))
	return false;
    }

  *inf_num = m_inf_num;

  bool thread_is_parsed = false;

  if (in_thread_state ())
      thread_is_parsed = process_thread_state (next);

  if (in_thread_state () && !thread_is_parsed)
    {
      /* Thread number was not parsed successfully and SIMD lanes are
	 not specified.  */
      m_state = STATE_INFERIOR;
      return false;
    }

  if (in_simd_lane_state ())
    {
      if (!process_simd_lane_state ())
	{
	  m_state = STATE_INFERIOR;
	  return false;
	}
    }
  else
    m_simd_lane_num = -1;

  *inf_num = m_inf_num;
  *thr_start = m_thr_num;

  if (simd_lane_num != nullptr)
    *simd_lane_num = m_simd_lane_num;

  /* If SIMD lane range is finished,  check if thread range is finished.  */
  if (!in_simd_lane_state () || (!m_simd_lane_range_parser.in_range ()
      && !m_simd_lane_range_parser.in_set ()))
    {
      /* If we successfully parsed a thread number or finished parsing a
	 thread range, switch back to assuming the next TID is
	 inferior-qualified.  */
      if (!m_range_parser.in_range () && !m_range_parser.in_set ())
	{
	  if (m_simd_lane_num == -1)
	    {
	      /* SIMD range was not specified.  */
	      m_cur_tok = m_range_parser.cur_tok ();
	    }
	  else
	    {
	      /* SIMD range was specified.  */
	      m_cur_tok = m_simd_lane_range_parser.cur_tok ();
	    }

	  m_state = STATE_INFERIOR;
	  m_in_thread_star_range = false;
	  m_in_simd_lane_star_range = false;

	  if (thr_end != nullptr)
	    *thr_end = *thr_start;
	}
      else
	{
	  /* Thread range is not yet finished.  Go back to the old thread
	     state.  */
	  m_state = STATE_THREAD_RANGE;
	}
    }

  /* If we're midway through a range, and the caller wants the end
     value, return it and skip to the end of the range.  */
  if (thr_end != nullptr && (in_thread_state () || in_simd_lane_state ()))
    {
      if (m_range_parser.in_range ())
	*thr_end = m_range_parser.end_value ();
      else
	*thr_end = *thr_start;

      skip_range ();
    }

  return true;
}

/* See tid-parse.h.  */

bool
tid_range_parser::get_tid_range (int *inf_num, int *thr_start, int *thr_end)
{
  gdb_assert (inf_num != NULL && thr_start != NULL && thr_end != NULL);

  return get_tid_or_range (inf_num, thr_start, thr_end, nullptr);
}

/* See tid-parse.h.  */

bool
tid_range_parser::get_tid (int *inf_num, int *thr_num, int *simd_lane_num)
{
  gdb_assert (inf_num != NULL && thr_num != NULL);

  return get_tid_or_range (inf_num, thr_num, nullptr, simd_lane_num);
}

/* See tid-parse.h.  */

bool
tid_range_parser::in_thread_star_range () const
{
  return (in_thread_state () || in_simd_lane_state ())
    && m_in_thread_star_range;
}

/* See tid-parse.h.  */

bool
tid_range_parser::in_simd_lane_star_range () const
{
  return in_simd_lane_state () && m_in_simd_lane_star_range;
}

/* See tid-parse.h.  */

int
tid_is_in_list (const char *list, int default_inferior,
		int inf_num, int thr_num)
{
  if (list == NULL || *list == '\0')
    return 1;

  tid_range_parser parser (list, default_inferior, 0);
  if (parser.finished ())
    invalid_thread_id_error (parser.cur_tok ());
  while (!parser.finished ())
    {
      int tmp_inf, tmp_thr_start, tmp_thr_end;

      if (!parser.get_tid_range (&tmp_inf, &tmp_thr_start, &tmp_thr_end))
	invalid_thread_id_error (parser.cur_tok ());
      if (tmp_inf == inf_num
	  && tmp_thr_start <= thr_num && thr_num <= tmp_thr_end)
	return 1;
    }
  return 0;
}

#if GDB_SELF_TEST
#include "gdbsupport/selftest.h"

namespace selftests {

/* Test 'tid_parser::get_tid ()'.  */

static void
test_get_tid ()
{
  /* Test parsing a SIMD lane range.  */
  {
    tid_range_parser parser {"1.3:1-3", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 2);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 3);
    SELF_CHECK (parser.finished ());
  }

  /* Test parsing a SIMD lane range put into brackets.  */
  {
    tid_range_parser parser {"1.3:[1-3]", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 2);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 3);
    SELF_CHECK (parser.finished ());
  }

  /* Test parsing a set of individual SIMD lanes.  */
  {
    tid_range_parser parser {"1.3:[2 4 6]", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 2);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 4);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 6);
    SELF_CHECK (parser.finished ());
  }

  /* Test parsing a thread range.  */
  {
	tid_range_parser parser {"1.1-3:1", 3, 5};
	int inf_num, thr_num, simd_lane_num;

	SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		    && inf_num == 1 && thr_num == 1 && simd_lane_num == 1);
	SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		    && inf_num == 1 && thr_num == 2 && simd_lane_num == 1);
	SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		    && inf_num == 1 && thr_num == 3 && simd_lane_num == 1);
	SELF_CHECK (parser.finished ());
  }

  /* Test parsing a thread range encapsulated into square brackets.  */
  {
    tid_range_parser parser {"1.[1-3]:1", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 1 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 2 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 1);
    SELF_CHECK (parser.finished ());
  }


  /* Test parsing a individual threads.  */
  {
    tid_range_parser parser {"1.[1 3 5]:1", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 1 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 3 && simd_lane_num == 1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 5 && simd_lane_num == 1);
    SELF_CHECK (parser.finished ());
  }


  /* Test parsing a individual threads.  */
  {
    tid_range_parser parser {"1.[4 7]:3-4", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 4 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 4 && simd_lane_num == 4);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 7 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 7 && simd_lane_num == 4);
    SELF_CHECK (parser.finished ());
  }


  /* Test parsing a individual threads.  */
  {
    tid_range_parser parser {"1.[4 7]:3-4 2.2-3:[7 8]", 3, 5};
    int inf_num, thr_num, simd_lane_num;

    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 4 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 4 && simd_lane_num == 4);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 7 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 1 && thr_num == 7 && simd_lane_num == 4);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 2 && thr_num == 2 && simd_lane_num == 7);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 2 && thr_num == 2 && simd_lane_num == 8);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 2 && thr_num == 3 && simd_lane_num == 7);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 2 && thr_num == 3 && simd_lane_num == 8);
    SELF_CHECK (parser.finished ());
  }

  /* Unqualified inferior.  */
  {
    tid_range_parser parser {"[2-3]:3-4 7.5-7 <command>", 3, 5};
    const char *cmd = "<command>";
    int inf_num, thr_num, simd_lane_num;

    /* First TID.  */
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 3 && thr_num == 2 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 3 && thr_num == 2 && simd_lane_num == 4);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 3 && thr_num == 3 && simd_lane_num == 3);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 3 && thr_num == 3 && simd_lane_num == 4);

    /* Second TID.  */
    simd_lane_num = 0;
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 7 && thr_num == 5 && simd_lane_num == -1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 7 && thr_num == 6 && simd_lane_num == -1);
    SELF_CHECK (parser.get_tid (&inf_num, &thr_num, &simd_lane_num)
		&& inf_num == 7 && thr_num == 7 && simd_lane_num == -1);

    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }
}

/* Test 'tid_parser::get_tid ()'.  */

static void
test_get_tid_range ()
{
  const char *cmd = "<command>";

  /* Thread range.  */
  {
    tid_range_parser parser {"1.4-7:3-4 <command>", 3, 5};
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 4 && thr_end == 7);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Thread range in square brackets.  */
  {
    tid_range_parser parser {"1.[4-7]:3-4 <command>", 3, 5};
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 4 && thr_end == 7);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Thread range with start == end.  */
  {
    tid_range_parser parser {"1.4-4:3-4 <command>", 3, 5};
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 4 && thr_end == 4);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Thread range with start == end, in square brackets.  */
  {
    tid_range_parser parser {"1.4-4:3-4 <command>", 3, 5};
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 4 && thr_end == 4);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Two individual threads.  */
  {
    tid_range_parser parser {"1.[2 4]:3-4 <command>", 3, 5};
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 2 && thr_end == 2);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Two tid expressions.  */
  {
    tid_range_parser parser {"1.[2-4]:3-4 3.[5-9] <command>", 3, 5};
    const char *expr = "3.[5-9] <command>";
    int inf_num, thr_start, thr_end;

    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 1 && thr_start == 2 && thr_end == 4);
    SELF_CHECK (strcmp (parser.cur_tok (), expr) == 0);
    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 3 && thr_start == 5 && thr_end == 9);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }

  /* Unqualified inferior.  */
  {
    tid_range_parser parser {"[2-4]:3-4 7.5-9 <command>", 3, 5};
    const char *expr = "7.5-9 <command>";
    int inf_num, thr_start, thr_end;

    SELF_CHECK (!parser.finished ());
    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 3 && thr_start == 2 && thr_end == 4);
    SELF_CHECK (!parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), expr) == 0);
    SELF_CHECK (parser.get_tid_range (&inf_num, &thr_start, &thr_end)
		&& inf_num == 7 && thr_start == 5 && thr_end == 9);
    SELF_CHECK (parser.finished ());
    SELF_CHECK (strcmp (parser.cur_tok (), cmd) == 0);
  }
}

}

#endif

void _initialize_tid_parse ();
void
_initialize_tid_parse ()
{
#if GDB_SELF_TEST
  selftests::register_test ("test_get_tid",
			    selftests::test_get_tid);
  selftests::register_test ("test_get_tid_range",
			    selftests::test_get_tid_range);
#endif
}
