/*******************************************************************************
* Copyright (C) 2003 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
!  Content:
!    viRngGeometric  Example Program Text
!******************************************************************************/

#include <stdio.h>
#include <math.h>

#include "mkl_vsl.h"
#include "errcheck.inc"

#define SEED    777
#define BRNG    VSL_BRNG_MCG31
#define METHOD  VSL_RNG_METHOD_GEOMETRIC_ICDF
#define N       1000
#define NN      10

int main(void)
{
    int r[N];
    VSLStreamStatePtr stream;
    int i, errcode = VSL_ERROR_OK, status = 0;
    double p = 0.5;

    double tM,tD,tQ,tD2;
    double sM,sD;
    double sum,sum2;
    double n,s;
    double DeltaM,DeltaD;

    /***** Initialize *****/
    errcode = vslNewStream( &stream, BRNG,  SEED );
    CheckVslError( errcode );

    /***** Call RNG *****/
    errcode = viRngGeometric( METHOD, stream, N, r, p );
    CheckVslError( errcode );

    /***** Theoretical moments *****/
    tM = (1.0 - p) / p;
    tD = (1.0 - p) / (p * p);
    tQ = (1.0 - p) * (p * p - 9.0 * p + 9.0) / (p * p * p * p);

    /***** Sample moments *****/
    sum = 0.0;
    sum2 = 0.0;
    for( i = 0; i < N; i++ )
    {
        sum += (double)r[i];
        sum2 += (double)r[i] * (double)r[i];
    }
    sM = sum / ((double)N);
    sD = sum2 / (double)N - (sM * sM);

    /***** Comparison of theoretical and sample moments *****/
    n = (double)N;
    tD2 = tD * tD;
    s = ((tQ - tD2) / n) - (2.0 * (tQ - 2.0 * tD2) / (n * n)) + ((tQ
        - 3.0 * tD2) / (n * n * n));

    DeltaM = (tM - sM) / sqrt(tD / n);
    DeltaD = (tD - sD) / sqrt(s);

    /***** Printing results *****/
    printf("Sample of viRngGeometric.\n");
    printf("-------------------------\n\n");
    printf("Parameters:\n");
    printf("    p=%.4f\n",p);

    printf("Results (first 10 of 1000):\n");
    printf("---------------------------\n");
    for( i = 0; i < NN; i++ )
    {
        printf("r[%d]=%d\n",i,r[i]);
    }

    printf("\n");
    if( fabs(DeltaM) > 3.0 || fabs(DeltaD) > 3.0 )
    {
        printf("Error: sample moments (mean=%.2f, variance=%.2f) disagree with theory (mean=%.2f, variance=%.2f).\n",sM,sD,tM,tD);
        status = 1;
    }
    else
    {
        printf("Sample moments (mean=%.2f, variance=%.2f) agree with theory (mean=%.2f, variance=%.2f).\n",sM,sD,tM,tD);
    }

    /***** Deinitialize *****/
    errcode = vslDeleteStream( &stream );
    CheckVslError( errcode );

    return status;
}
